﻿/*
 * Copyright (c) 2013, M@T
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the <organization> nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL M@T BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.IO;
using System.Windows.Forms;

namespace LibertyTicket_Distro_ROM_Editor
{
    public partial class frm_Header : Form
    {
        private uint romLogoAndTextOffset = 0x3C400;
        private const uint ChecksumOffset = 0x2;
        private const uint BlockStartOffset = 0x20;
        private const int BlockSize = 0x820;
        private const uint TextStartOffset = 0x240;
        private const int TextSize = 0x100;

        private uint currentChar;
        private int previousLangID = 2;

        private bool formLoaded = false;

        private MemoryStream tempMs;

        public frm_Header()
        {
            InitializeComponent();
        }

        private uint getRomLogoAndTextOffset()
        {
            BinaryReader br = new BinaryReader(frm_Main.ms);
            br.BaseStream.Position = 0x68;
            return br.ReadUInt32();
        }

        private void refreshData()
        {
            frm_Main.ms.Position = TextSize * cb_Lang.SelectedIndex + romLogoAndTextOffset + TextStartOffset;

            tb_Text.Text = "";

            for (int i = 0; i < TextSize / 2; i++)
            {
                currentChar = Convert.ToUInt32(frm_Main.ms.ReadByte() + frm_Main.ms.ReadByte() * 0x100);

                if (currentChar == 0x0000)
                    break;

                else if (currentChar == 0x000A)
                    tb_Text.Text += "\r\n";

                else
                    tb_Text.Text += Convert.ToChar(currentChar);
            }
        }

        private void saveText(int langID)
        {
            tempMs.Position = TextSize * langID + TextStartOffset - BlockStartOffset;

            for (int i = 0; i < TextSize; i++) // efface tout le texte en le remplaçant par des 0x00
                tempMs.WriteByte(0x00);

            tempMs.Position = TextSize * langID + TextStartOffset - BlockStartOffset;

            for (int i = 0; i < tb_Text.Text.Length; i++)
            {
                if (tb_Text.Text[i] == '\r' && tb_Text.Text[i + 1] == '\n') // retour à la ligne
                {
                    tempMs.WriteByte(0x0A);
                    tempMs.WriteByte(0x00);
                    i++; // on incrémente pour sauter le '\n'
                }
                else
                {
                    tempMs.WriteByte(Convert.ToByte((int)(tb_Text.Text[i] & 0xFF)));
                    tempMs.WriteByte(Convert.ToByte((int)(tb_Text.Text[i] >> 8)));
                }
            }
        }

        private ushort crc16(byte[] data)
        {
            ushort crc = 0xFFFF, currVal, tabVal;
            ushort[] seedTable = { 0x0000, 0xCC01, 0xD801, 0x1400, 0xF001, 0x3C00, 0x2800, 0xE401, 0xA001, 0x6C00, 0x7800, 0xB401, 0x5000, 0x9C01, 0x8801, 0x4400 };

            for (int i = 0; i < data.Length; i += 2)
            {
                currVal = (ushort)(data[i] + data[i + 1] * 0x100);

                for (int j = 0; j < 4; j++)
                {
                    tabVal = seedTable[crc & 0xF];
                    crc >>= 4;
                    crc ^= tabVal;
                    crc ^= seedTable[(currVal >> (4 * j)) & 0xF];
                }
            }

            return crc;
        }

        private void frm_Header_Load(object sender, EventArgs e)
        {
            romLogoAndTextOffset = getRomLogoAndTextOffset();
            cb_Lang.SelectedIndex = 2;

            byte[] buffer = new byte[BlockSize];
            frm_Main.ms.Position = romLogoAndTextOffset + BlockStartOffset;
            frm_Main.ms.Read(buffer, 0, BlockSize);
            tempMs = new MemoryStream(buffer);

            refreshData();

            formLoaded = true;
        }

        /*private void Form2_FormClosed(object sender, FormClosedEventArgs e)
        {
            this.Dispose(true);
        }*/

        private void tb_Text_TextChanged(object sender, EventArgs e)
        {
            int crLfCnt = (tb_Text.Text.Length - tb_Text.Text.Replace("\r\n", "").Length) / 2;

            tb_Text.MaxLength = 128 + crLfCnt;

            lbl_CharCnt.Text = (tb_Text.Text.Length - crLfCnt).ToString() + "/128";
        }

        private void comboBox1_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (formLoaded)
            {
                saveText(previousLangID);
                previousLangID = cb_Lang.SelectedIndex;
                refreshData();
            }
        }

        private void b_Validate_Click(object sender, EventArgs e)
        {
            saveText(cb_Lang.SelectedIndex);

            tempMs.Position = 0;
            byte[] buffer = new byte[BlockSize];
            tempMs.Read(buffer, 0, BlockSize);

            frm_Main.ms.Position = romLogoAndTextOffset + BlockStartOffset;
            frm_Main.ms.Write(buffer, 0, BlockSize);

            ushort checksum = crc16(buffer);

            frm_Main.ms.Position = romLogoAndTextOffset + ChecksumOffset;
            frm_Main.ms.WriteByte((byte)(checksum & 0xFF));
            frm_Main.ms.WriteByte((byte)(checksum >> 8));

            this.Close();
        }

        private void b_Cancel_Click(object sender, EventArgs e)
        {
            this.Close();
        }
    }
}
